﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using Microsoft.Win32;
using System.IO;
using System.Windows.Markup;

namespace OcdToXamlConverter {

	public abstract partial class OcdBaseFileReader {

		public OcdBaseFileReader() {
			isLoaded = false;
		}

		private bool isLoaded;

		public bool IsLoaded {
			get {
				return isLoaded;
			}
			set {
				isLoaded = value;
			}
		}

		public abstract int GetMainVersion();

		public abstract int GetSubVersion();

		public abstract double GetScale();

		public string GetVersionString() {
			if (!IsLoaded) {
				return "";
			}
			return GetMainVersion().ToString() + "." + GetSubVersion().ToString();
		}

		public abstract FrameworkElement CreateMap();

		public abstract void CleanUp();

		public abstract bool LoadFile(Stream stream);

		protected abstract Color GetColor(Int16 num);

		protected static int GetMainColorNum(TSymElt[] elts) {
			if (elts.Length == 0) {
				return -1;
			}
			return elts[0].stColor;
		}

		protected static Geometry GetAreaGeometry(TCord[] cords) {
			if (cords.Length == 0) {
				return null;
			}
			PathGeometry geometry = null;
			PathFigure pathFigure;
			PathGeometry pathGeometry;
			PathSegmentCollection pathSegmentCollection;
			Point firstPoint;
			pathFigure = new PathFigure();
			pathGeometry = new PathGeometry(new PathFigure[] { pathFigure });
			pathSegmentCollection = new PathSegmentCollection();
			pathFigure.Segments = pathSegmentCollection;
			firstPoint = new Point(GetLength(cords[0].XPos), -GetLength(cords[0].YPos));
			pathFigure.StartPoint = firstPoint;
			bool inHole = false;
			int i;
			for (i = 1; i < cords.Length; i++) {
				if (cords[i].IsFirstHolePoint) {
					if (inHole) {
						geometry = Geometry.Combine(geometry, pathGeometry, GeometryCombineMode.Xor, null);
					} else {
						geometry = pathGeometry;
					}
					pathFigure = new PathFigure();
					pathGeometry = new PathGeometry(new PathFigure[] { pathFigure });
					pathSegmentCollection = new PathSegmentCollection();
					pathFigure.Segments = pathSegmentCollection;
					pathFigure.StartPoint = new Point(GetLength(cords[i].XPos), -GetLength(cords[i].YPos));
					inHole = true;
				} else {
					if (cords[i].IsFirstCurvePoint) {
						if (i >= cords.Length - 2) {
							break;
						}
						BezierSegment bezierSegment = new BezierSegment(new Point(GetLength(cords[i].XPos), -GetLength(cords[i].YPos)), new Point(GetLength(cords[i + 1].XPos), -GetLength(cords[i + 1].YPos)), new Point(GetLength(cords[i + 2].XPos), -GetLength(cords[i + 2].YPos)), false);
						pathSegmentCollection.Add(bezierSegment);
						i += 2;
					} else {
						LineSegment lineSegment = new LineSegment();
						lineSegment.Point = new Point(GetLength(cords[i].XPos), -GetLength(cords[i].YPos));
						pathSegmentCollection.Add(lineSegment);
					}
				}
			}
			if (inHole) {
				geometry = Geometry.Combine(geometry, pathGeometry, GeometryCombineMode.Xor, null);
			} else {
				geometry = pathGeometry;
			}
			return geometry;
		}

		protected static Rect GetExtent(TSymElt[] elts) {
			Rect rect = new Rect();
			if (elts.Length == 0) {
				return rect;
			}
			rect = GetExtent(elts[0]);
			for (int i = 1; i < elts.Length; i++) {
				rect = Rect.Union(rect, GetExtent(elts[i]));
			}
			return rect;
		}

		protected static Rect GetExtent(TSymElt elt) {
			Rect rect = new Rect();
			if (elt == null) {
				return rect;
			}
			if (elt.stPoly.Length <= 0) {
				return rect;
			}
			PathFigure pathFigure;
			Geometry geometry;
			double diameter;
			switch (elt.stType) {
				case 1:
					OcdPathHelper pathHelper = new OcdPathHelper(elt.stPoly);
					pathFigure = pathHelper.GetPathFigure();
					geometry = new PathGeometry(new PathFigureCollection(new PathFigure[] { pathFigure }));
					Pen pen = new Pen(Brushes.Black, GetLength(elt.stLineWidth));
					rect = geometry.GetRenderBounds(pen);
					return rect;
				case 2:
					geometry = GetAreaGeometry(elt.stPoly);
					rect = geometry.Bounds;
					return rect;
				case 3:
					diameter = GetLength(elt.stDiameter);
					rect = new Rect(GetLength(elt.stPoly[0].XPos) - diameter / 2, -GetLength(elt.stPoly[0].YPos) - diameter / 2, diameter, diameter);
					return rect;
				case 4:
					diameter = GetLength(elt.stDiameter);
					rect = new Rect(GetLength(elt.stPoly[0].XPos) - diameter / 2, -GetLength(elt.stPoly[0].YPos) - diameter / 2, diameter, diameter);
					return rect;
			}
			return rect;
		}

		protected abstract int GetColorDepth(int num);

		protected static double BaseLengthUnit = 96 / 2.54 * 0.001;

		protected static double GetLength(double num) {
			return num * BaseLengthUnit;
		}

		private int SortElts(TSymElt elt1, TSymElt elt2) {
			int num1 = GetColorDepth(elt1.stColor);
			int num2 = GetColorDepth(elt2.stColor);
			return num1.CompareTo(num2);
		}

		protected DrawingGroup CreateElementsGroup(TSymElt[] elts, bool subtractStrokeWidth) {
			if (elts.Length == 0) {
				return null;
			}
			List<TSymElt> eltsList = new List<TSymElt>(elts);
			eltsList.Sort(SortElts);
			DrawingGroup group = new DrawingGroup();
			for (int i = 0; i < eltsList.Count; i++) {
				Drawing elt = CreateElement(eltsList[i], subtractStrokeWidth);
				if (elt != null) {
					group.Children.Add(elt);
				}
			}
			return group;
		}

		protected Drawing CreateElement(TSymElt elt, bool subtractStrokeWidth) {
			Color color = GetColor(elt.stColor);
			Brush brush = new SolidColorBrush(color);
			GeometryDrawing drawing;
			Geometry geometry;
			PathFigure pathFigure;
			switch (elt.stType) {
				case 1:
					OcdPathHelper pathHelper = new OcdPathHelper(elt.stPoly);
					pathFigure = pathHelper.GetPathFigure();
					geometry = new PathGeometry(new PathFigureCollection(new PathFigure[] { pathFigure }));
					drawing = new GeometryDrawing(null, new Pen(brush, GetLength(elt.stLineWidth)), geometry);
					return drawing;
				case 2:
					geometry = GetAreaGeometry(elt.stPoly);
					drawing = new GeometryDrawing(brush, null, geometry);
					return drawing;
				case 3:
					if (elt.stPoly.Length <= 0) {
						return null;
					}
					if (subtractStrokeWidth) {
						geometry = new EllipseGeometry(elt.stPoly[0].GetPoint(), GetLength(elt.stDiameter) / 2 - GetLength(elt.stLineWidth) / 2, GetLength(elt.stDiameter) / 2 - GetLength(elt.stLineWidth) / 2);
					} else {
						geometry = new EllipseGeometry(elt.stPoly[0].GetPoint(), GetLength(elt.stDiameter) / 2, GetLength(elt.stDiameter) / 2);
					}
					drawing = new GeometryDrawing(null, new Pen(brush, GetLength(elt.stLineWidth)), geometry);
					return drawing;
				case 4:
					geometry = new EllipseGeometry(elt.stPoly[0].GetPoint(), GetLength(elt.stDiameter) / 2, GetLength(elt.stDiameter) / 2);
					drawing = new GeometryDrawing(brush, null, geometry);
					return drawing;
				default:
					return null;
			}
		}

		protected FrameworkElement CreatePointSymbol(TSymElt[] elts, bool subtractStrokeWidth) {
			Drawing drawing = CreateElementsGroup(elts, subtractStrokeWidth);
			if (drawing == null) {
				return null;
			}
			CustomDrawingContainer container = new CustomDrawingContainer();
			container.Drawing = drawing;
			return container;
		}

		protected class TSymbolBlock {

			public Int32 NextBlock;
			public Int32[] FilePos = new Int32[256];

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				NextBlock = br.ReadInt32();
				for (int i = 0; i < FilePos.Length; i++) {
					FilePos[i] = br.ReadInt32();
				}
				return true;
			}

		}

		protected class TSymElt {

			public Int16 stType;
			public UInt16 stFlags; //word
			public Int16 stColor;
			public Int16 stLineWidth;
			public Int16 stDiameter;
			public Int16 stnPoly;
			public Int16 stRes1;
			public Int16 stRes2;
			public TCord[] stPoly;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				stType = br.ReadInt16();
				stFlags = br.ReadUInt16();
				stColor = br.ReadInt16();
				stLineWidth = br.ReadInt16();
				stDiameter = br.ReadInt16();
				stnPoly = br.ReadInt16();
				stRes1 = br.ReadInt16();
				stRes2 = br.ReadInt16();
				stPoly = new TCord[stnPoly];
				for (int i = 0; i < stPoly.Length; i++) {
					TCord cord = new TCord();
					cord.Load(stream);
					stPoly[i] = cord;
				}
				return true;
			}

		}

		protected class TCord {

			public Int32 x;
			public Int32 y;

			public bool Load(Stream stream) {
				BinaryReader br = new BinaryReader(stream);
				x = br.ReadInt32();
				y = br.ReadInt32();
				return true;
			}

			//Custom Members

			public Int32 XPos {
				get {
					return x >> 8;
				}
			}
			public Int32 YPos {
				get {
					return y >> 8;
				}
			}
			public bool IsCurvePoint {
				get {
					if (IsFirstCurvePoint || IsSecondCurvePoint) {
						return true;
					}
					return false;
				}
			}
			public bool IsFirstCurvePoint {
				get {
					int flags = x << 24;
					flags = flags >> 24;
					if ((flags & 1) == 1) {
						return true;
					}
					return false;
				}
			}
			public bool IsSecondCurvePoint {
				get {
					int flags = x << 24;
					flags = flags >> 24;
					if ((flags & 2) == 2) {
						return true;
					}
					return false;
				}
			}
			public bool IsFirstHolePoint {
				get {
					int flags = y << 24;
					flags = flags >> 24;
					if ((flags & 2) == 2) {
						return true;
					}
					return false;
				}
			}
			public bool IsCornerPoint {
				get {
					int flags = y << 24;
					flags = flags >> 24;
					if ((flags & 1) == 1) {
						return true;
					}
					return false;
				}
			}
			public bool IsLeftStroked {
				get {
					int flags = x << 24;
					flags = flags >> 24;
					if ((flags & 4) == 4) {
						return false;
					}
					return true;
				}
			}
			public bool IsRightStroked {
				get {
					int flags = y << 24;
					flags = flags >> 24;
					if ((flags & 4) == 4) {
						return false;
					}
					return true;
				}
			}

			public Point GetPoint() {
				return new Point(OcdBaseFileReader.GetLength(XPos), -OcdBaseFileReader.GetLength(YPos));
			}

		}

	}

	public class OcdFileReader {

		private OcdBaseFileReader reader;

		public OcdFileReader() {
			reader = null;
		}

		public void CleanUp() {
			if (reader != null) {
				reader.CleanUp();
			}
		}

		public bool LoadFile(string fileName) {
			CleanUp();
			FileStream fs = new FileStream(fileName, FileMode.Open);
			BinaryReader br = new BinaryReader(fs);
			Int16 ocadMark = br.ReadInt16();
			if (ocadMark != 0x0cad) {
				return false;
			}
			Int16 ignore = br.ReadInt16();
			Int16 version = br.ReadInt16();
			fs.Seek(0, SeekOrigin.Begin);
			if (version >= 6 && version <= 8) {
				reader = new Ocd8FileReader();
				return reader.LoadFile(fs);
			} else if (version == 9 || version == 10) {
				reader = new Ocd9FileReader();
				return reader.LoadFile(fs);
			}
			return false;
		}

		public int GetMainVersion() {
			if (reader == null) {
				return 0;
			}
			return reader.GetMainVersion();
		}

		public int GetSubVersion() {
			if (reader == null) {
				return 0;
			}
			return reader.GetSubVersion();
		}

		public string GetVersionString() {
			if (reader == null) {
				return "";
			}
			return reader.GetVersionString();
		}

		public double GetScale() {
			if (reader == null) {
				return 0;
			}
			return reader.GetScale();
		}

		public FrameworkElement CreateMap() {
			if (reader == null) {
				return null;
			}
			return reader.CreateMap();
		}

		public static bool SaveAsXaml(UIElement elt, string fileName) {
			FileStream fs = new FileStream(fileName, FileMode.Create);
			XamlWriter.Save(elt, fs);
			fs.Close();
			return true;
		}

	}

}
