﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;
using System.Windows.Xps;
using System.Windows.Xps.Packaging;
using Microsoft.Win32;
using System.IO;
using System.IO.Packaging;
using System.Printing;
using System.Windows.Markup;
using System.Collections.ObjectModel;

namespace OPlanner {

	public partial class ExportPrintDescriptions : Window {

		private OCurrentInfo info;
		private DescriptionManager descriptionManager;
		private PrintDialog printDialog;
		private ExportDescriptionsCollection coursesItems;

		public ExportPrintDescriptions() {
			InitializeComponent();
			coursesItems = new ExportDescriptionsCollection();
			dpiBox.Text = "96";
			qualityBox.Text = "80";
		}

		public void SetPrintDialog(PrintDialog newDialog) {
			printDialog = newDialog;
			UpdatePrintInfo();
		}

		private void UpdatePrintInfo() {
			printerLabel.Content = "Printer: " + printDialog.PrintQueue.FullName;
		}

		public void SetInfo(OCurrentInfo newInfo) {
			info = newInfo;
			descriptionSizeBox.Text = Convert.ToString(OPlannerHelper.PixelsToMillimetres(info.Data.DescriptionSize));
			UpdateCourses();
		}

		public void SetDescriptionManager(DescriptionManager newManager) {
			descriptionManager = newManager;
		}


		private bool Validate(bool showErrors) {
			ErrorManager errorManager = new ErrorManager();
			double num;
			if (descriptionSizeBox.Text == "") {
				errorManager.AddError("Description size is required");
			} else {
				if (!double.TryParse(descriptionSizeBox.Text, out num)) {
					errorManager.AddError("Description size must be a number");
				} else {
					if (num <= 0) {
						errorManager.AddError("Description size must be a positive number");
					}
				}
			}
			if (showErrors) {
				errorManager.ShowErrors(this);
			}
			return !errorManager.HasErrors;
		}

		private bool ValidateExport(bool showErrors) {
			ErrorManager errorManager = new ErrorManager();
			double num;
			if (dpiBox.Text == "") {
				errorManager.AddError("Dpi is required");
			} else {
				if (!double.TryParse(dpiBox.Text, out num)) {
					errorManager.AddError("Dpi must be a number");
				} else {
					if (num <= 0 || num > 600) {
						errorManager.AddError("Dpi must be between 1 and 600");
					}
				}
			}
			if (jpegButton.IsChecked == true) {
				int val;
				if (qualityBox.Text == "") {
					errorManager.AddError("Quality is required");
				} else {
					if (!int.TryParse(qualityBox.Text, out val)) {
						errorManager.AddError("Quality must be an integer");
					} else {
						if (val <= 0 || val > 100) {
							errorManager.AddError("Quality must be between 1 and 100");
						}
					}
				}
			}
			if (showErrors) {
				errorManager.ShowErrors(this);
			}
			return !errorManager.HasErrors;
		}

		private void exportButton_Click(object sender, RoutedEventArgs e) {
			if (!Validate(true)) {
				return;
			}
			if (!ValidateExport(true)) {
				return;
			}
			double dpi = double.Parse(dpiBox.Text);
			int quality = int.Parse(qualityBox.Text);
			double boxSize = OPlannerHelper.MillimetresToPixels(double.Parse(descriptionSizeBox.Text));
			string title = titleBox.Text;
			if (coursesItems.HasSelectedItems == false) {
				ErrorManager errorManager = new ErrorManager();
				errorManager.AddError("Please select a course");
				errorManager.ShowErrors(this);
				return;
			}
			for (int i = 0; i < coursesItems.Count; i++) {
				double borderWidth = 10;
				if (coursesItems[i].ExportSymbol == true) {
					Course objectData = coursesItems[i].Course;
					FrameworkElement descriptions;
					if (objectData == null) {
						descriptions = descriptionManager.GetDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
					}
					else {
						descriptions = descriptionManager.GetDescriptions(objectData, Brushes.Black, boxSize, 0, title);
					}
					descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
					descriptions.Arrange(new Rect(new Point(borderWidth, borderWidth), descriptions.DesiredSize));
					Size size = descriptions.DesiredSize;
					SaveFileDialog dlg = new SaveFileDialog();
					dlg.FileName = coursesItems[i].CourseName.Replace(" ", "") + "Symbol";
					if (jpegButton.IsChecked == true) {
						dlg.DefaultExt = ".jpg";
						dlg.FileName += ".jpg";
						dlg.Filter = "Jpeg|*.jpg";
					} else if (pngButton.IsChecked == true) {
						dlg.DefaultExt = ".png";
						dlg.FileName += ".png";
						dlg.Filter = "Png|*.png";
					}
					if (dlg.ShowDialog(this) == true) {
						RenderTargetBitmap bmp = new RenderTargetBitmap((int)(Math.Ceiling((size.Width + borderWidth * 2)*dpi/96.0)), (int)(Math.Ceiling((size.Height + borderWidth * 2)*dpi/96.0)), dpi, dpi, PixelFormats.Pbgra32);
						BitmapEncoder encoder;
						if (jpegButton.IsChecked == true) {
							encoder = new JpegBitmapEncoder();
							((JpegBitmapEncoder)encoder).QualityLevel = quality;
						}
						else if (pngButton.IsChecked == true) {
							encoder = new PngBitmapEncoder();
						}
						else {
							continue;
						}
						Rectangle rect = new Rectangle();
						rect.Width = bmp.Width;
						rect.Height = bmp.Height;
						rect.Fill = Brushes.White;
						rect.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
						rect.Arrange(new Rect(new Point(0, 0), rect.DesiredSize));
						bmp.Render(rect);
						bmp.Render(descriptions);
						encoder.Frames.Add(BitmapFrame.Create(bmp));
						string fileName = dlg.FileName;
						FileStream fs = new FileStream(fileName, FileMode.Create);
						encoder.Save(fs);
						fs.Close();
					}
				}
				if (coursesItems[i].ExportText == true) {
					Course objectData = coursesItems[i].Course;
					FrameworkElement descriptions;
					if (objectData == null) {
						descriptions = descriptionManager.GetTextDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
					}
					else {
						descriptions = descriptionManager.GetTextDescriptions(objectData, Brushes.Black, boxSize, 0, title);
					}
					descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
					descriptions.Arrange(new Rect(new Point(borderWidth, borderWidth), descriptions.DesiredSize));
					Size size = descriptions.DesiredSize;
					SaveFileDialog dlg = new SaveFileDialog();
					dlg.FileName = coursesItems[i].CourseName.Replace(" ", "") + "Text";
					if (jpegButton.IsChecked == true) {
						dlg.DefaultExt = ".jpg";
						dlg.FileName += ".jpg";
						dlg.Filter = "Jpeg|*.jpg";
					} else if (pngButton.IsChecked == true) {
						dlg.DefaultExt = ".png";
						dlg.FileName += ".png";
						dlg.Filter = "Png|*.png";
					}
					if (dlg.ShowDialog(this) == true) {
						RenderTargetBitmap bmp = new RenderTargetBitmap((int)(Math.Ceiling((size.Width + borderWidth * 2) * dpi / 96.0)), (int)(Math.Ceiling((size.Height + borderWidth * 2) * dpi / 96.0)), dpi, dpi, PixelFormats.Pbgra32);
						BitmapEncoder encoder;
						if (jpegButton.IsChecked == true) {
							encoder = new JpegBitmapEncoder();
							((JpegBitmapEncoder)encoder).QualityLevel = quality;
						}
						else if (pngButton.IsChecked == true) {
							encoder = new PngBitmapEncoder();
						}
						else {
							continue;
						}
						Rectangle rect = new Rectangle();
						rect.Width = bmp.Width;
						rect.Height = bmp.Height;
						rect.Fill = Brushes.White;
						rect.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
						rect.Arrange(new Rect(new Point(0, 0), rect.DesiredSize));
						bmp.Render(rect);
						bmp.Render(descriptions);
						encoder.Frames.Add(BitmapFrame.Create(bmp));
						string fileName = dlg.FileName;
						FileStream fs = new FileStream(fileName, FileMode.Create);
						encoder.Save(fs);
						fs.Close();
					}
				}
			}
		}

		private void UpdateCourses() {
			coursesView.ItemsSource = null;
			if (info == null) {
				return;
			}
			if (info.Data == null) {
				return;
			}
			List<Course> tempList = new List<Course>(info.Data.Courses);
			tempList.Sort(CompareCourses);
			coursesItems.Clear();
			coursesItems.Add(new ExportDescriptionsItem(null));
			for (int i = 0; i < tempList.Count; i++) {
				ExportDescriptionsItem item = new ExportDescriptionsItem(tempList[i]);
				coursesItems.Add(item);
			}
			coursesView.ItemsSource = coursesItems;
		}

		private static int CompareCourses(Course x, Course y) {
			return x.Name.CompareTo(y.Name);
		}

		private void previewButton_Click(object sender, RoutedEventArgs e) {
			if (coursesItems.HasSelectedItems == false) {
				ErrorManager errorManager = new ErrorManager();
				errorManager.AddError("Please select a course");
				errorManager.ShowErrors(this);
				return;
			}
			if (!Validate(true)) {
				return;
			}
			FixedDocument fixedDoc = GetFixedDocument();
			if (fixedDoc == null) {
				return;
			}
			PrintPreview wnd = new PrintPreview();
			wnd.Owner = this;
			wnd.SetPrintDialog(printDialog);
			wnd.SetFixedDoc(fixedDoc);
			wnd.SetPrintName("OPlanner Descriptions");
			wnd.ShowDialog();
		}

		private FixedDocument GetFixedDocument() {
			if (!Validate(false)) {
				return null;
			}
			string title = titleBox.Text;
			double boxSize = OPlannerHelper.MillimetresToPixels(double.Parse(descriptionSizeBox.Text));
			if (coursesItems.HasSelectedItems == false) {
				return null;
			}
			PrintCapabilities capabilities = printDialog.PrintQueue.GetPrintCapabilities(printDialog.PrintTicket);
			Size pageSize = new Size(printDialog.PrintableAreaWidth, printDialog.PrintableAreaHeight);
			Size visibleSize = new Size(capabilities.PageImageableArea.ExtentWidth, capabilities.PageImageableArea.ExtentHeight);
			FixedDocument fixedDoc = new FixedDocument();
			PageContent pageContent = new PageContent();
			FixedPage page = new FixedPage();
			((IAddChild)pageContent).AddChild(page);
			fixedDoc.Pages.Add(pageContent);
			page.Width = pageSize.Width;
			page.Height = pageSize.Height;
			Canvas canvas = new Canvas();
			FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
			FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
			canvas.Width = visibleSize.Width;
			canvas.Height = visibleSize.Height;
			page.Children.Add(canvas);
			double borderWidth = 10;
			double x = borderWidth;
			double y = borderWidth;
			double maxHeight = 0;
			if (coursesItems.NumSelectedCourses == 1) {
				ExportDescriptionsItem item = coursesItems.SelectedItems[0];
				if (onePerPage.IsChecked == true) {
					bool descriptionAdded = false;
					if (item.ExportSymbol == true) {
						Course objectData = item.Course;
						FrameworkElement descriptions;
						if (objectData == null) {
							descriptions = descriptionManager.GetDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
						}
						else {
							descriptions = descriptionManager.GetDescriptions(objectData, Brushes.Black, boxSize, 0, title);
						}
						descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
						Size descriptionSize = descriptions.DesiredSize;
						Canvas.SetLeft(descriptions, x);
						Canvas.SetTop(descriptions, y);
						canvas.Children.Add(descriptions);
						descriptionAdded = true;
					}
					if (item.ExportText == true) {
						Course objectData = item.Course;
						FrameworkElement descriptions;
						if (objectData == null) {
							descriptions = descriptionManager.GetTextDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
						}
						else {
							descriptions = descriptionManager.GetTextDescriptions(objectData, Brushes.Black, boxSize, 0, title);
						}
						descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
						Size descriptionSize = descriptions.DesiredSize;
						if (descriptionAdded) {
							if (descriptionAdded) {
								pageContent = new PageContent();
								page = new FixedPage();
								((IAddChild)pageContent).AddChild(page);
								fixedDoc.Pages.Add(pageContent);
								page.Width = pageSize.Width;
								page.Height = pageSize.Height;
								canvas = new Canvas();
								FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
								FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
								canvas.Width = visibleSize.Width;
								canvas.Height = visibleSize.Height;
								page.Children.Add(canvas);
								y = borderWidth;
								x = borderWidth;
							}
						}
						Canvas.SetLeft(descriptions, x);
						Canvas.SetTop(descriptions, y);
						canvas.Children.Add(descriptions);
						descriptionAdded = true;
					}
				} else {
					bool descriptionAdded = false;
					for (int i = 0; i < 50; i++) {
						if (item.ExportSymbol == true) {
							Course objectData = item.Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (!(descriptionSize.Width + borderWidth * 2 > visibleSize.Width)) {
								if (x + descriptionSize.Width + borderWidth > visibleSize.Width) {
									y += maxHeight + borderWidth;
									x = borderWidth;
									maxHeight = 0;
								}
							}
							if (descriptionAdded) {
								if (descriptionSize.Height + borderWidth + y > visibleSize.Height) {
									break;
								}
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							x += descriptionSize.Width + borderWidth;
							if (descriptionSize.Height > maxHeight) {
								maxHeight = descriptionSize.Height;
							}
							descriptionAdded = true;
						}
						if (item.ExportText == true) {
							Course objectData = item.Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetTextDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetTextDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (!(descriptionSize.Width + borderWidth * 2 > visibleSize.Width)) {
								if (x + descriptionSize.Width + borderWidth > visibleSize.Width) {
									y += maxHeight + borderWidth;
									x = borderWidth;
									maxHeight = 0;
								}
							}
							if (descriptionAdded) {
								if (descriptionSize.Height + borderWidth + y > visibleSize.Height) {
									break;
								}
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							x += descriptionSize.Width + borderWidth;
							if (descriptionSize.Height > maxHeight) {
								maxHeight = descriptionSize.Height;
							}
							descriptionAdded = true;
						}
					}
				}
			} else {
				if (onePerPage.IsChecked == true) {
					bool descriptionAdded = false;
					for (int i = 0; i < coursesItems.Count; i++) {
						if (coursesItems[i].ExportSymbol == true) {
							Course objectData = coursesItems[i].Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (descriptionAdded) {
								pageContent = new PageContent();
								page = new FixedPage();
								((IAddChild)pageContent).AddChild(page);
								fixedDoc.Pages.Add(pageContent);
								page.Width = pageSize.Width;
								page.Height = pageSize.Height;
								canvas = new Canvas();
								FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
								FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
								canvas.Width = visibleSize.Width;
								canvas.Height = visibleSize.Height;
								page.Children.Add(canvas);
								y = borderWidth;
								x = borderWidth;
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							descriptionAdded = true;
						}
						if (coursesItems[i].ExportText == true) {
							Course objectData = coursesItems[i].Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetTextDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetTextDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (descriptionAdded) {
								pageContent = new PageContent();
								page = new FixedPage();
								((IAddChild)pageContent).AddChild(page);
								fixedDoc.Pages.Add(pageContent);
								page.Width = pageSize.Width;
								page.Height = pageSize.Height;
								canvas = new Canvas();
								FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
								FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
								canvas.Width = visibleSize.Width;
								canvas.Height = visibleSize.Height;
								page.Children.Add(canvas);
								y = borderWidth;
								x = borderWidth;
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							descriptionAdded = true;
						}
					}
				}
				else {
					for (int i = 0; i < coursesItems.Count; i++) {
						if (coursesItems[i].ExportSymbol == true) {
							Course objectData = coursesItems[i].Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (!(descriptionSize.Width + borderWidth * 2 > visibleSize.Width)) {
								if (x + descriptionSize.Width + borderWidth > visibleSize.Width) {
									y += maxHeight + borderWidth;
									x = borderWidth;
									maxHeight = 0;
								}
							}
							if (!(descriptionSize.Height + borderWidth * 2 > visibleSize.Height)) {
								if (y + descriptionSize.Height + borderWidth > visibleSize.Height) {
									pageContent = new PageContent();
									page = new FixedPage();
									((IAddChild)pageContent).AddChild(page);
									fixedDoc.Pages.Add(pageContent);
									page.Width = pageSize.Width;
									page.Height = pageSize.Height;
									canvas = new Canvas();
									FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
									FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
									canvas.Width = visibleSize.Width;
									canvas.Height = visibleSize.Height;
									page.Children.Add(canvas);
									y = borderWidth;
									x = borderWidth;
								}
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							x += descriptionSize.Width + borderWidth;
							if (descriptionSize.Height > maxHeight) {
								maxHeight = descriptionSize.Height;
							}
						}
						if (coursesItems[i].ExportText == true) {
							Course objectData = coursesItems[i].Course;
							FrameworkElement descriptions;
							if (objectData == null) {
								descriptions = descriptionManager.GetTextDescriptions(info.Data, Brushes.Black, boxSize, 0, title);
							}
							else {
								descriptions = descriptionManager.GetTextDescriptions(objectData, Brushes.Black, boxSize, 0, title);
							}
							descriptions.Measure(new Size(double.PositiveInfinity, double.PositiveInfinity));
							Size descriptionSize = descriptions.DesiredSize;
							if (!(descriptionSize.Width + borderWidth * 2 > visibleSize.Width)) {
								if (x + descriptionSize.Width + borderWidth > visibleSize.Width) {
									y += maxHeight + borderWidth;
									x = borderWidth;
									maxHeight = 0;
								}
							}
							if (!(descriptionSize.Height + borderWidth * 2 > visibleSize.Height)) {
								if (y + descriptionSize.Height + borderWidth > visibleSize.Height) {
									pageContent = new PageContent();
									page = new FixedPage();
									((IAddChild)pageContent).AddChild(page);
									fixedDoc.Pages.Add(pageContent);
									page.Width = pageSize.Width;
									page.Height = pageSize.Height;
									canvas = new Canvas();
									FixedPage.SetLeft(canvas, capabilities.PageImageableArea.OriginWidth);
									FixedPage.SetTop(canvas, capabilities.PageImageableArea.OriginHeight);
									canvas.Width = visibleSize.Width;
									canvas.Height = visibleSize.Height;
									page.Children.Add(canvas);
									y = borderWidth;
									x = borderWidth;
								}
							}
							Canvas.SetLeft(descriptions, x);
							Canvas.SetTop(descriptions, y);
							canvas.Children.Add(descriptions);
							x += descriptionSize.Width + borderWidth;
							if (descriptionSize.Height > maxHeight) {
								maxHeight = descriptionSize.Height;
							}
						}
					}
				}
			}
			return fixedDoc;
		}

		private void printerButton_Click(object sender, RoutedEventArgs e) {
			PageMediaSizeName? currentPageMediaSize = printDialog.PrintTicket.PageMediaSize.PageMediaSizeName;
			PageOrientation? currentPageOrientation = printDialog.PrintTicket.PageOrientation;
			printDialog.ShowDialog();
			if (printDialog.PrintTicket.PageMediaSize.PageMediaSizeName != currentPageMediaSize || printDialog.PrintTicket.PageOrientation != currentPageOrientation) {
				info.InvokeDataChanging(this, "Change Print Descriptions Settings");
				if (printDialog.PrintTicket.PageMediaSize.PageMediaSizeName == printDialog.PrintQueue.DefaultPrintTicket.PageMediaSize.PageMediaSizeName) {
					info.Data.DescriptionsPageMediaSize = null;
				} else {
					info.Data.DescriptionsPageMediaSize = printDialog.PrintTicket.PageMediaSize.PageMediaSizeName;
				}
				if (printDialog.PrintTicket.PageOrientation == printDialog.PrintQueue.DefaultPrintTicket.PageOrientation) {
					info.Data.DescriptionsPageOrientation = null;
				} else {
					info.Data.DescriptionsPageOrientation = printDialog.PrintTicket.PageOrientation;
				}
				info.InvokeDataChanged(this, "Change Print Descriptions Settings");
			}
			UpdatePrintInfo();
		}

		private void jpegButton_Checked(object sender, RoutedEventArgs e) {
			if (qualityPanel != null) {
				qualityPanel.Visibility = Visibility.Visible;
			}
		}

		private void jpegButton_Unchecked(object sender, RoutedEventArgs e) {
			if (qualityPanel != null) {
				qualityPanel.Visibility = Visibility.Collapsed;
			}
		}

	}

	public class ExportDescriptionsItem {

		public ExportDescriptionsItem() {
			exportText = false;
			exportSymbol = false;
		}

		public ExportDescriptionsItem(Course course) : this() {
			this.course = course;
		}

		private Course course;
		public Course Course {
			get {
				return course;
			}
			set {
				course = value;
			}
		}

		public string CourseName {
			get {
				if (Course == null) {
					return "All Controls";
				}
				return Course.Name;
			}
		}

		private bool exportText;
		public bool ExportText {
			get {
				return exportText;
			}
			set {
				exportText = value;
			}
		}

		private bool exportSymbol;
		public bool ExportSymbol {
			get {
				return exportSymbol;
			}
			set {
				exportSymbol = value;
			}
		}

	}

	public class ExportDescriptionsCollection : ObservableCollection<ExportDescriptionsItem> {

		public bool HasSelectedItems {
			get {
				for (int i = 0; i < Count; i++) {
					if (this[i].ExportSymbol == true || this[i].ExportText == true) {
						return true;
					}
				}
				return false;
			}
		}

		public int NumSelectedCourses {
			get {
				int num = 0;
				for (int i = 0; i < Count; i++) {
					if (this[i].ExportSymbol == true || this[i].ExportText == true) {
						num++;
					}
				}
				return num;
			}
		}

		public List<ExportDescriptionsItem> SelectedItems {
			get {
				List<ExportDescriptionsItem> list = new List<ExportDescriptionsItem>();
				for (int i = 0; i < Count; i++) {
					if (this[i].ExportSymbol == true || this[i].ExportText == true) {
						list.Add(this[i]);
					}
				}
				return list;
			}
		}

	}

}
